/*********************************************************************
*	 项目名称:MG82F6P32-DEMO
*			测试板使用 TH253A MG82F6P32_LQFP32_QFN32_EV_V11 EV Board
*			CpuCLK=12MHz, SysCLK=12MHz
*	功能描述：
*			读取16字节 UID,	并利用它对程序加密
*			 使用U1/U2 脱机烧录系列号,程序判断该系列号是否在正确的范围
*	注意事项：
*
*	 建立时间: 2024.2.26
*	 修改日志:
*
*********************************************************************/
#define	_MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>	// for printf

#include ".\include\REG_MG82F6P32.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6P32.H"
#include ".\include\API_Uart_BRGRL_MG82F6P32.H"

/*************************************************
*定义系统时钟SysClk (MAX.50MHz)
*可选:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define	MCU_SYSCLK		12000000
/*************************************************/
/*************************************************
*定义CPU时钟 CpuClk	(MAX.36MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define	MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define	TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256)
#define	TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define	TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	/256)
#define	TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	%256)

#define	IO_LED_R		P22
#define	IO_LED_Y		P24
#define	IO_LED_G		P26

#define	ISP_ADDRESS	   0x7E00	 //ISP起始地址
#define	IAP_ADDRESS	   0x7600	 //IAP起始地址

#define	UID_ENCRYPT_BUF_ADDRESS	0x1000
#define	DEVICE_SN_BUF_ADDRESS	0x0200


code u8	UIDEncryptBuf[16] _at_ UID_ENCRYPT_BUF_ADDRESS;

// 设置系列号,可以使用U1/U2 脱机烧录系列号，起始地址设为下面系列号变量定义的地址（如: 0x0200）
// 如果该系列号,不在正确的范围内,则可能烧录文件被盗用
code u8	DeviceSn[16] _at_ DEVICE_SN_BUF_ADDRESS;

// U1/U2 脱机烧录的系列号为12为字符+4字节数据
// 在这里判断12位字符是否吻合
code u8	DeviceSnString[]="Megawin--MCU";

/*************************************************
*函数名称: char	putchar	(char c)
*功能描述: 用于printf	的串口输出程序
*输入参数:	   char	c
*返回参数:
*************************************************/
char putchar (char c)
{
	bit	bES;
	bES=ES0;
	ES0=0;
	S0BUF =	c;
	while(TI0==0);
	TI0=0;
	ES0=bES;
	return 0;
}

/*************************************************
*函数名称:	   void	DelayXus(u16 xUs)
*功能描述:		延时程序，单位为us
*输入参数:	   u8 Us ->	*1us  (1~255)
*返回参数:
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if	(MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if	(MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if	(MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*函数名称:	   void	DelayXms(u16 xMs)
*功能描述:	   延时程序，单位为ms
*输入参数:	   u16 xMs -> *1ms	(1~65535)
*返回参数:
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;

	}
}

/***********************************************************************************
*函数名称:	 void InitUart0_S0BRG(void)
*功能描述:	 Uart0初始化 使用S0BRG作为波特率来源
*输入参数:
*返回参数:
*************************************************************************************/
void InitUart0_S0BRG(void)
{
	UART0_SetMode8bitUARTVar();			// 8位 可变波特率
	UART0_EnReception();				// 使能接收
	UART0_SetBRGFromS0BRG();			// 设置波特率来源为	S0BRG
	UART0_SetS0BRGBaudRateX2();			// 设置2x波特率
	UART0_SetS0BRGSelSYSCLK();			// 设置S0BRG时钟输入为Sysclk
	UART0_SetRxTxP30P31();
	// Sets	B.R. value
	UART0_SetS0BRGValue(S0BRG_BRGRL_9600_2X_12000000_1T);

	UART0_EnS0BRG();					// 开启S0BRG时钟
}

/***********************************************************************************
*函数名称:	 void InitPort(void)
*功能描述:	 IO配置
*输入参数:
*返回参数:
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2PushPull(BIT2|BIT4|BIT6);		// 设置P22,P24,P26为推挽输出(LED)
	PORT_SetP3QuasiBi(BIT0|BIT1);					// 设置P30,P31 作为准双向口
}

/***********************************************************************************
*函数名称:	 void InitClock(void)
*功能描述:	 时钟初始化
*输入参数:
*返回参数:
*************************************************************************************/
void InitClock(void)
{
#if	(MCU_SYSCLK==11059200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==12000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=12MHz	CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==22118400)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==24000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==29491200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==32000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz	CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz	CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if	(MCU_SYSCLK==48000000)
	// SysClk=48MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 输出 MCK/4
	//CLK_P60OC_MCKDiv4();
}


/***********************************************************************************
*函数名称:	 u8	IAP_ReadPPage(u8 PsfrAddr)
*功能描述:读取P页特殊功能器内容
*输入参数:
*		 u8	PsfrAddr: P页地址
*返回参数:
*		 u8: P页内容
*************************************************************************************/
u8 IAP_ReadPPage(u8	PsfrAddr)
{
	bit	bEA=EA;
	EA = 0;						//关中断
	IFADRH = 0;					//IFADRH必须为0
	IFADRL=	PsfrAddr;			//送P页地址;
	IFMT = ISP_READ_P;
	ISPCR =	0x80;				//设置等待时间, 允许ISP/IAP操作
	SCMD = 0x46;
	SCMD = 0xB9;
	nop();
	IFMT=0;
	ISPCR =	0;					//清ISP/IAP特殊寄存器，防止误操作
	EA = bEA;					//恢复中断
	return IFD;
}

/***********************************************************************************
*函数名称:	 void IAP_WritePPage(u8	PsfrAddr,u8	PsfrData)
*功能描述:写P页特殊功能器
*输入参数:
*		 u8	PsfrAddr: P页地址
*		 u8	PsfrData: P页内容
*返回参数:
*************************************************************************************/
void IAP_WritePPage(u8 PsfrAddr,u8 PsfrData)
{
	bit	bEA=EA;
	EA = 0;						//关中断
	IFADRH = 0;					//IFADRH必须为0
	IFADRL=	PsfrAddr;			//送P页地址;
	IFD= PsfrData;				//送P页内容
	IFMT = ISP_WRITE_P;
	ISPCR =	0x80;				//设置等待时间, 允许ISP/IAP操作
	SCMD = 0x46;
	SCMD = 0xB9;
	nop();
	IFMT=0;
	ISPCR =	0;					//清ISP/IAP特殊寄存器，防止误操作
	EA = bEA;					//恢复中断
}


/***********************************************************************************
*函数名称:u8 IAP_ReadByte(u16 ByteAddr)
*功能描述:读指定地址的IAP内容
*输入参数:
*		 u16 ByteAddr: IAP地址
*返回参数:
*************************************************************************************/
u8 IAP_ReadByte(u16	ByteAddr)
{
		bit	bEA=EA;
		IFADRH = ByteAddr>>8;		//送地址高字节
		IFADRL=	ByteAddr;			//送地址低字节;
		EA = 0;						//关中断
		IFMT = ISP_READ;
		ISPCR =	0x80;				//允许ISP/IAP操作
		SCMD = 0x46;
		SCMD = 0xB9;
		IFMT=0;
		ISPCR =	0;					//清ISP/IAP特殊寄存器，防止误操作
		EA = bEA;					//恢复中断
		return IFD;
}
// 可以使用MOVC的方式读取IAP的数据
#define	IAP_ReadByteByMOVC(x)	CBYTE[x]

/***********************************************************************************
*函数名称:void IAP_ErasePage(u8	ByteAddr)
*功能描述:页面擦除,	擦除指定地址所在的页面。
*		擦除一个页面大约需要30ms
*输入参数: u8 ByteAddr:	指定的页面地址高8位
*返回参数:
*************************************************************************************/
void IAP_ErasePage(u8 ByteAddr)
{
		bit	bEA=EA;
		IFADRH = ByteAddr;		//送地址高字节
		IFADRL=	0x00;			//送地址低字节;
		EA = 0;					//关中断
		IFMT = ISP_ERASE;
		ISPCR =	0x80;			//允许ISP/IAP操作
		SCMD = 0x46;
		SCMD = 0xB9;
		nop();
		IFMT=0;
		ISPCR =	0;					//清ISP/IAP特殊寄存器，防止误操作
		EA = bEA;					//恢复中断
}

/***********************************************************************************
*函数名称:void IAP_WriteByte(u16 ByteAddr,u8 ByteData)
*功能描述:写字节到指定地址,	该地址必须为空(0xFF),否则要先执行页面擦除
*		 写入一个字节数据大约需要80us
*输入参数:
*		 u16 ByteAddr: 指定的IAP地址, u8	ByteData: 待写的数据
*返回参数:
*************************************************************************************/
void IAP_WriteByte(u16 ByteAddr,u8 ByteData)
{
		bit	bEA=EA;
		IFD	= ByteData;				//送所要写的数据
		IFADRH = ByteAddr>>8;		//送地址高字节
		IFADRL=	ByteAddr;			//送地址低字节;
		EA = 0;						//关中断
		IFMT = ISP_WRITE;
		ISPCR =	0x80;				//允许ISP/IAP操作
		SCMD = 0x46;
		SCMD = 0xB9;
		nop();
		IFMT=0;
		ISPCR =	0;					//清ISP/IAP特殊寄存器，防止误操作
		EA = bEA;					//恢复中断
}

/***********************************************************************************
*函数名称:	 void IAP_ReadUID(u8 *pUID)
*功能描述:读取UID
*输入参数: u8 *PUID: 存储16Byte UID的缓存的指针
*返回参数:
*************************************************************************************/
void IAP_ReadUID(u8	*pUID)
{
	u8 i;
	ISPCR =	0x80;
	BOREV =	0x22;
	IFMT = 0x06;
	IFADRH = 0x00;
	IFADRL = 0xF0;

	for(i =	0; i < 16; i ++)
	{
		SCMD = 0x46;
		SCMD = 0xB9;
		pUID[i]	= IFD;
		IFADRL ++;
	}
	BOREV =	0x00;
	IFMT=0;
	ISPCR =	0x00;
}

/***********************************************************************************
*函数名称:	 u8	UID_Encrypt()
*功能描述:使用UID(唯一ID)进行加密检测
*输入参数:
*返回参数:0:Pass, other:Fail
*************************************************************************************/
u8 UID_Encrypt(void)
{
	xdata u8 UID[16];
	u8 i;

	IAP_ReadUID(&UID[0]);

	for(i=0;i<16;i++)
	{
		if(UIDEncryptBuf[i]!=0xFF) break;
	}
	if(i<16)
	{ // 存储有加密字节, 比对是否正确
		for(i=0;i<16;i++)
		{
			if(UID[i] != UIDEncryptBuf[i]) break;
		}
		if(i<16)
		{ // 比对错误
			return i+1;
		}
		else
		{
			return 0;
		}
	}
	else
	{ // 全为FF,表示未存储加密字节
		IAP_WritePPage(IAPLB_P,UID_ENCRYPT_BUF_ADDRESS/256);
		for(i=0;i<16;i++)
		{
			IAP_WriteByte(UID_ENCRYPT_BUF_ADDRESS+i, UID[i]);
		}
		IAP_WritePPage(IAPLB_P,ISP_ADDRESS/256);
		return 0;
	}

	return 0;


}

/***********************************************************************************/

/***********************************************************************************
*函数名称:	 void InitSystem(void)
*功能描述:	 系统初始化
*输入参数:
*返回参数:
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitClock();
	InitUart0_S0BRG();
}

void main()
{

	u8 i;
	bit	bUIDEncryptFlag,bDeviceStringFlag;

	InitSystem();

	printf("\nHello!!");

	IO_LED_G=0;IO_LED_Y=0;IO_LED_R=0;
	DelayXms(1000);
	IO_LED_G=1;IO_LED_Y=1;IO_LED_R=1;

	// 判断唯一ID是否吻合
	if(UID_Encrypt()!=0)
	{ // 否
		printf("\nUID Test Fail!!");
		bUIDEncryptFlag=FALSE;
		IO_LED_R=0;
	}
	else
	{ // 是
		printf("\nUID Test Pass!!");
		bUIDEncryptFlag=TRUE;
	}

	// 判断脱机序列号是否吻合
	bDeviceStringFlag=TRUE;
	for(i=0;i<12;i++)
	{
		if(DeviceSnString[i] !=	DeviceSn[i])
		{ //否
			bDeviceStringFlag=FALSE;
		}
	}

	if(bDeviceStringFlag)
		{
			printf("\nDeviceSN Test	Pass!!");
		}
		else
		{
			printf("\nDeviceSN Test	Fail!!");
			IO_LED_Y=0;
		}

	while(1)
	{
		IO_LED_G=!IO_LED_G;
		DelayXms(100);
	}

}

